package es.aeat.pret.c200.c231.api.srv;

import java.io.Serializable;
import java.util.List;

import es.aeat.pret.c200.api.bean.ErrorValidacionBean;
import es.aeat.pret.c200.api.bean.PerceptorBean;



/**
 * Interfaz del servicio de clculo de retenciones 2020 para un perceptor
 * individual, cuyos datos estarn almacenados en un objeto {@link PerceptorBean}.
 * 
 * La forma normal de trabajar con este servicio es:
 * <ol>
 * 		<li> Instanciar un objeto <code>CalculoRetencionesSrv</code> mediante la correspondiente factora {@link PRET_C220_Factory}
 * 		<li> Crear un nuevo objeto {@link PerceptorBean}
 * 		<li> Asignar valores al objeto {@code PerceptorBean} mediante los mtodos <code>setValor</code>
 * 		<li> Validar
 * 		<li> Si la validacin es correcta, ejecutar el clculo y extraer los resultados. En caso contrario, recoger la lista de errores.
 * </ol>
 *
 * Por ejemplo:
 * <pre>
 *	//Instanciar el servicio
 *	CalculoRetencionesSrv calculo = PRET_C220_Factory.getCalculoRetencionesSrv();
 *	// Crear un objeto percepto
 *	PerceptorBean perceptor = calculo.createPerceptorBean();
 *	//Asignar valores al bean
 *	perceptor.setValor(PerceptorBean.NIFPER, "00000101D");
 *	perceptor.setValor(PerceptorBean.APEPER, "JUAN ESPAOL ESPAOL");
 *		.
 *		.
 *		.
 *	perceptor.setValor(PerceptorBean.RETRIB, "50000.00");
 *	perceptor.setValor(PerceptorBean.COTIZACIONES, "1000.00");
 *
 *	// Validar
 *	if(calculo.validar(perceptor)){
 *		// Calcular
 *		calculo.calcular(perceptor);
 *		// Obtener resultados
 *		out.println("IMPORTE ANUAL DE LAS RETENCIONES E INGRESOS A CUENTA "+perceptor.getValor("IMPORTE"));
 *		out.println(" TIPO DE RETENCIN APLICABLE "+perceptor.getValor("TIPO"));
 *	}
 *	else {
 *		// Recoger la lista de errores.
 *		List<String[]> errores = calculo.getErrores(perceptor);
 *		int numErr=errores.size();
 *		int numErr=errores.size();
 *		out.println("TENEMOS "+numErr+" ERRORES");out.println("\n");
 *		List<String[]> errores = calculo.getErrores(perceptor);
 *		for(String[] error : errores){
 *		   out.println("CODIGO "+error[0]);out.println("\n");
 *		   out.println("TEXTO "+error[1]);out.println("\n");
 *		}
 *	}
 * </pre>
 */
public interface CalculoRetencionesSrv extends Serializable {
	/**
	 * Crea un objeto {@link PerceptorBean} vaco.
	 * 
	 * @return el nuevo objeto {@link PerceptorBean} 
	 */
	PerceptorBean createPerceptorBean();
	
	/**
	 * Ejecuta las validaciones de los datos contenidos en el <i>bean</i>
	 * segn lo establecido en el <a href="http://www.agenciatributaria.es/static_files/AEAT/Contenidos_Comunes/La_Agencia_Tributaria/Informacion_institucional/Campanias/Retenciones_trabajo_personal/2015/ALGORITMO2015.pdf">algoritmo de clculo</a>
	 * del tipo de retencin para el ejercicio 2015.
	 * 
	 * @param perceptorBean objeto <code>PerceptorBean</a> que vamos a validar
	 * @param desdeFichero indica si vamos a validar para un clculo desde fichero  XML.
	 * @return <code>true</code> Si las validacin es correcta<br/>
	 * 		   <code>false</false> Si existe algn error. En este caso se puede recuperar la lista de errores mediante
	 * 			el mtodo {@link #getErrores}
	 */
	boolean validar(PerceptorBean perceptorBean, boolean desdeFichero);
	
	/**
	 * Ejecuta los clculos sobre los datos establecidos en el <i>bean</i> 
	 * segn lo establecido en el <a href="http://www.agenciatributaria.es/static_files/AEAT/Contenidos_Comunes/La_Agencia_Tributaria/Informacion_institucional/Campanias/Retenciones_trabajo_personal/2015/ALGORITMO2015.pdf">algoritmo de clculo</a>.
	 * El resultado queda en los datos del propio <i>bean</i>.
	 * 
	 * @param perceptorBean el objeto <code>PerceptorBean</a> que vamos a calcular.
	 */
	void calcular(PerceptorBean perceptorBean);
	
	
	/**
	 * Obtiene la lista de errores producidos al validar el objeto {@link PerceptorBean}.
	 * 
	 * @param perceptorBean el objeto <code>PerceptorBean</code> que hemos validado.
	 * @return la lista de errores producidos. Cada error se devuelve como una lista de objetos <code>ErrorValidacionBean</code>. 
	 * 
	 */
	List<ErrorValidacionBean> getErrores(PerceptorBean perceptorBean);

	/**
	 * Indica si se tiene que dar el aviso de monoparental
	 * @param perceptorBean
	 * @return
	 */
	boolean avisoMonoparental(PerceptorBean perceptorBean);
	
	/**
	 * Indica si se tienen que dar el aviso de TIPOA
	 * @param perceptorBean
	 * @return
	 */
	boolean avisoTipoA(PerceptorBean perceptorBean);

	/**
	 * Indica si se tienen que mostrar los avisos relativos a BASEA
	 * @param perceptorBean
	 * @return
	 */
	boolean avisosBaseA(PerceptorBean perceptorBean);

	/**
	 * Indica si se tienen que mostrar los avisos relativos a RETRIBA
	 * @param perceptorBean
	 * @return
	 */
	boolean avisosRetribA(PerceptorBean perceptorBean);

	/**
	 * Realiza las ltimas validaciones posteriores a los clculos
	 * @param perceptorBean
	 * @return
	 */
	boolean validarPostCalculos(PerceptorBean perceptorBean);
	

	/**
	 * Devuelve la versin del mdulo de clculo.
	 * 
	 * @return la versin.
	 */
	String getVersion();

	/**
	 * Construye un objeto PerceptorBean a partir de la representacin de
	 * JSON.
	 * 
	 * @param beanJson el JSON del bean a construir
	 * @return el objeto PerceptorBean equivalente al JSON recibido.
	 */
	PerceptorBean createPerceptorBeanFromJson(String beanJson); 

}
