package es.aeat.pret.c200.mc.c210;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Paths;
import java.util.ResourceBundle;
import java.util.logging.LogManager;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;

import es.aeat.pret.c200.c210.api.srv.CalculoRetencionesXMLSrv;
import es.aeat.pret.c200.c210.imp.srv.CalculoRetencionesXMLSrvImpl;
import es.aeat.pret.c200.mc.comun.AnalizarLineaComandoException;
import es.aeat.pret.c200.mc.comun.AnalizarLineaComandos;
import es.aeat.pret.c200.mc.comun.PathTrabajo;

/**
 * Clase utilidad usada como punto de entrada al API para lanzar el proceso de
 * un fichero XML para el clculo de Retenciones 2015.
 * 
 * En funcin de la forma en que se disponga del documento de entrada se deber
 * invocar a alguna de las diferentes funciones procesarFicheroXml
 * 
 * Ampliacin del API.
 * 
 * @author DIT - Agencia Tributaria
 */
public class ModuloCalculo {
	private static final String NO_SE_HA_BORRADO = "No se pudo borrar el fichero: %s";
//	private static org.slf4j.Logger logger;

	/*
	 * Clase utilidad. No se pueden instanciar objetos de esta clase.
	 */
	private ModuloCalculo() {
	}

	/* ************************** API Pblica **************************** */

	/**
	 * Procesa un fichero XML de retenciones
	 * 
	 * @param ficheroEntrada nombre del fichero XML de entrada
	 * @param ficheroErrores nombre del fichero XML de salida de Errores
	 * @param flag           Se ignora
	 * @param ficheroSalida  nombre del fichero XML de salida de Resultados
	 */
	public static void procesarFicheroXml(String ficheroEntrada, String ficheroErrores, String flag, String ficheroSalida) {
		// Siempre se establece el flag de mdulo de clculo en la clase del 2020
		es.aeat.pret.c200.mc.ModuloCalculo.setModuloCalculo(true);

		borrarFichero(ficheroErrores);
		
		borrarFichero(ficheroSalida);

		File file = new File(ficheroEntrada); 
		if (!file.exists()) {
			try {
				generarErrorEntradaNoExiste(file);
			} catch (FileNotFoundException e) {
				getLogger().error("Imposible crear fichero de errores.");
			}
			return;
		}

		CalculoRetencionesXMLSrv modulo = null;

		String copiaEntrada = null;

		String pathResultado = null;

		try {
			copiaEntrada = copyFile(ficheroEntrada, null);
			modulo = new CalculoRetencionesXMLSrvImpl();
			pathResultado = modulo.procesar(new File(copiaEntrada), null);
			if (pathResultado.contains("SRET")) {
				borrarFichero(ficheroSalida);
				copyFile(pathResultado, ficheroSalida);
			} else {
				borrarFichero(ficheroErrores);
				copyFile(pathResultado, ficheroErrores);
			}
		} catch (Exception e) {
			getLogger().error(e.getMessage(), e);
		} finally {
			borrarFichero(copiaEntrada);
			borrarFichero(pathResultado);
		}
	}

	private static void borrarFichero(String pathFichero) {
		try {
			Files.delete(Paths.get(pathFichero));
		} catch (NoSuchFileException e) {
			getLogger().trace(String.format(NO_SE_HA_BORRADO, pathFichero));
		} catch (IOException e) {
			getLogger().error(String.format(NO_SE_HA_BORRADO, pathFichero), e);
		}
	}



	/**
	 * 
	 * Procesa un documento XML de retenciones de entrada.
	 * 
	 * @param entradaXML Cadena con el documento de entrada.
	 * @return Una cadena con un documento xml de salida si el fichero es correcto o un
	 *         documento XML de error.
	 * @throws SAXException
	 * @throws IOException
	 */

	@SuppressWarnings("resource")
	public static String procesarFicheroXML(String entradaXML) throws IOException {
		// Siempre se establece el flag de mdulo de clculo en la clase del 2020
		es.aeat.pret.c200.mc.ModuloCalculo.setModuloCalculo(true);

		ReadableByteChannel sourceChannel = null;
		FileChannel destChannel = null;
		CalculoRetencionesXMLSrv mod = null;

		File dest = null;
		String pathResultado = null;

		try {
			// copiamos el documento xml recibido a un ficheo temporal
			dest = File.createTempFile("stringfile-", null);

			destChannel = new FileOutputStream(dest).getChannel(); // NOSONAR

			byte[] bytesEntrada = entradaXML.getBytes();
			sourceChannel = Channels.newChannel(new ByteArrayInputStream(bytesEntrada));

			destChannel.transferFrom(sourceChannel, 0, bytesEntrada.length);

			try {
				mod = new CalculoRetencionesXMLSrvImpl();
				pathResultado = mod.procesar(dest, null);
				return fileToString(pathResultado);
			} catch (Exception e) {
				getLogger().error(e.getMessage(), e);
			}
		} finally {
			if (sourceChannel != null) {
				sourceChannel.close();
			}
			if (destChannel != null) {
				destChannel.close();
			}

			if (dest != null && !dest.delete()) { // NOSONAR
				getLogger().error("No se pudo borrar el fichero " + dest.getAbsolutePath()); // NOSONAR
			}

			if (pathResultado != null && !new File(pathResultado).delete()) {
				getLogger().error(NO_SE_HA_BORRADO);
			}
		}
		return null;
	}

	/* ************************** FIN API Pblica **************************** */

	private static void generarErrorEntradaNoExiste(File errores) throws FileNotFoundException  {
		try (PrintWriter writer = new PrintWriter(errores)) {
			writer.println("<?xml version=\"1.0\" encoding=\"ISO-8859-1\" standalone=\"yes\"?>");
			writer.println("<AEATRetencionesError2021>");
			writer.println("<ErrorGeneral>");
			writer.println("<Linea>0</Linea>");
			writer.println("<Posicion>0</Posicion>");
			writer.println("<Descripcion>El fichero de entrada no existe.</Descripcion>");
			writer.println("</ErrorGeneral>");
			writer.println("</AEATRetencionesError2021>");
			writer.flush();
		} 
	}

	@SuppressWarnings("resource")
	private static String fileToString(String origen) throws IOException {

		FileChannel sourceChannel = null;
		WritableByteChannel destChannel = null;
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		try {
			File source = new File(origen);
			sourceChannel = new FileInputStream(source).getChannel(); // NOSONAR
			destChannel = Channels.newChannel(baos); // NOSONAR
			ByteBuffer buffer = ByteBuffer.allocateDirect(16 * 1024);
			while (sourceChannel.read(buffer) != -1) {
				buffer.flip();
				destChannel.write(buffer);
				buffer.compact();
			}
			buffer.flip();
			while (buffer.hasRemaining()) {
				destChannel.write(buffer);
			}

			return baos.toString();
		} finally {
			if (sourceChannel != null) {
				sourceChannel.close();
			}
			if (destChannel != null) {
				destChannel.close();
			}
		}
	}

	@SuppressWarnings("resource")
	private static String copyFile(String origen, String destino) throws IOException {
		FileChannel sourceChannel = null;
		FileChannel destChannel = null;
		try {
			File source = new File(origen);
			sourceChannel = new FileInputStream(source).getChannel(); // NOSONAR
			File dest;
			if (destino == null) {
				dest = File.createTempFile("copyOf-" + source.getName() + "-", null);
				dest.deleteOnExit();
			} else {
				dest = new File(destino);
			}

			destChannel = new FileOutputStream(dest).getChannel(); // NOSONAR
			destChannel.transferFrom(sourceChannel, 0, sourceChannel.size());
			return dest.getPath();
		} finally {
			if (sourceChannel != null) {
				sourceChannel.close();
			}
			if (destChannel != null) {
				destChannel.close();
			}
		}
	}

	/**
	 * Mtodo {@linkplain main} para la ejecucin del m&oacute;dulo de
	 * c&aacute;lculo como una aplicaci&oacute;n independiente.
	 * 
	 * @exclude
	 */
	public static void main(String[] args) {
		// Siempre se establece el flag de mdulo de clculo en la clase del 2020
		es.aeat.pret.c200.mc.ModuloCalculo.setModuloCalculo(true);

		configLog();

		AnalizarLineaComandos alc = new AnalizarLineaComandos();
		alc.setParametro("E", true, null, true);
		alc.setParametro("R", true, null, true);
		alc.setParametro("S", true, null, true);
		alc.setParametro("I", false, "null", false);
		String fichero = null;
		try {
			alc.parse(args);
			fichero = alc.getParametro("E");
			String errores = alc.getParametro("R");
			String flag = "";

			String salida = alc.getParametro("S");

			String adicional = alc.getParametro("I");

			if (!"null".equals(adicional)) {
				ResourceBundle bundle = ResourceBundle.getBundle("es/aeat/pret/c200/mc/c210/retenciones");
				System.out.println(bundle.getString("acercade.titulo") + " " + bundle.getString("acercade.periodo") + " " + bundle.getString("version.mc")); // NOSONAR
			}

			procesarFicheroXml(fichero, errores, flag, salida);
		} catch (AnalizarLineaComandoException ex) {
			getLogger().error("Error en lnea de comandos", ex);
			System.exit(1);
		}
	}

	private static void configLog() {
		try {
			LogManager logManager = LogManager.getLogManager();

			InputStream prop = ModuloCalculo.class.getResourceAsStream("/es/aeat/pret/c200/mc/comun/logging.properties");
			logManager.readConfiguration(prop);
			prop.close();

			// Si se ejecuta el mdulo de clculo como un programa
			// Intento meter dinmicamente la implementacin de logging
			// java en el classpath
			File file = new File(PathTrabajo.getPathTrabajoSinProtocolo(), "slf4j-jdk14-1.6.1.jar");
			if (file.exists()) {
				addURL(file.toURI().toURL());
			}
		} catch (Exception e) {
			getLogger().error("Error, could not add URL to system classloader", e);
		}
	}

	private static void addURL(URL u) throws NoSuchMethodException, IllegalAccessException, InvocationTargetException {
		URLClassLoader sysloader = (URLClassLoader) ClassLoader.getSystemClassLoader();
		Class<URLClassLoader> sysclass = URLClassLoader.class;

		Method method = sysclass.getDeclaredMethod("addURL", URL.class);
		method.setAccessible(true);
		method.invoke(sysloader, u);
	}

	public static Logger getLogger() {
		return LoggerFactory.getLogger(ModuloCalculo.class);
	}
}
