package es.aeat.pret.c200.imp.srv;

import java.io.IOException;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import es.aeat.adht.jdit.api.ADHT_JDIT_Factory;
import es.aeat.adht.jdit.api.infra.ContextoSrv;
import es.aeat.adht.util.api.ADHT_UTIL_Factory;
import es.aeat.adht.util.api.utilidades.DitDataInputStreamBean;
import es.aeat.adht.util.api.utilidades.DitDataOutputStreamBean;
import es.aeat.pret.c200.api.PRET_C200_Factory;
import es.aeat.pret.c200.api.bean.AscendienteBean;
import es.aeat.pret.c200.api.bean.DescendienteBean;
import es.aeat.pret.c200.api.bean.ErrorValidacionBean.TipoErrorValidacion;
import es.aeat.pret.c200.api.bean.PerceptorBean;
import es.aeat.pret.c200.api.bean.PerceptorBeanTipos;
import es.aeat.pret.c200.api.srv.CalculoRetencionesCICSSrv;
import es.aeat.pret.c200.api.srv.CalculoRetencionesSrv;
import es.aeat.pret.c200.imp.bean.ErrorValidacionBeanImpl;
import es.aeat.pret.c200.util.ValidaNif;


/**
 * Implementacin del servicio {@link CalculoRetencionesCICSSrv}
 * 
 */
public class CalculoRetencionesCICSSrvImpl implements CalculoRetencionesCICSSrv {

	
	@Override
	public DitDataOutputStreamBean procesaJDIT(DitDataInputStreamBean datos) throws Throwable {
		PerceptorBean perceptorBean = leerBeanDeComarea(datos);
		
		if (perceptorBean.getErr().isEmpty()) {
			CalculoRetencionesSrv srv = PRET_C200_Factory.getCalculoRetencionesSrv();
			
			if (srv.validar(perceptorBean, true)) {
				srv.calcular(perceptorBean);
				traza("Resultado:\n\t" + "Tipo: " + perceptorBean.getValor(PerceptorBeanTipos.P_TIPO) + "\n\t"+ "Importe: " + perceptorBean.getValor(PerceptorBeanTipos.P_IMPORTE));
			} else {
				traza(perceptorBean.getErr().stream().map(e -> e.getCodigo() + " " + e.getMensaje() + "\n\t").reduce("Errores de validacin en clculo:\n\t", String::concat));
			}
		} else {
			traza(perceptorBean.getErr().stream().map(e -> e.getCodigo() + " " + e.getMensaje() + "\n\t").reduce("Errores de validacin en registro 190:\n\t" , String::concat));
		}
		return getComareaSalida(perceptorBean);
	}

	private DitDataOutputStreamBean getComareaSalida(PerceptorBean perceptorBean) throws IOException {
		final DitDataOutputStreamBean outputStreamBean = ADHT_UTIL_Factory.getDOSSrv().getDOS();
		
		if (perceptorBean.getErr().isEmpty()) {
			outputStreamBean.writeBoolean(true);
			outputStreamBean.writeDouble((perceptorBean.getValorAsBigDecimal(PerceptorBeanTipos.P_TIPO)).doubleValue());
			outputStreamBean.writeDouble((perceptorBean.getValorAsBigDecimal(PerceptorBeanTipos.P_IMPORTE)).doubleValue());
		}
		else {
			outputStreamBean.writeBoolean(false);
			String listaErrores = perceptorBean.getErr().stream().map(e -> e.getCodigo() + ";").reduce("", String::concat);
			listaErrores = listaErrores.substring(0, listaErrores.length() - 1);
			outputStreamBean.writeString(listaErrores, listaErrores.length());
		}
		
		return outputStreamBean;
	}

	private PerceptorBean leerBeanDeComarea(DitDataInputStreamBean datos) throws IOException {
		byte[] b = new byte[500];
		
		datos.read(b);
		String registro = new String(b);
		final PerceptorBean perceptorBean = PRET_C200_Factory.getCalculoRetencionesSrv().createPerceptorBean();
		
		registroABean(perceptorBean, registro);
		
		return perceptorBean;
	}

	private void registroABean(PerceptorBean perceptorBean, String registro) {
		List<ErrorValidacionBeanImpl> erroresRegistro = new ArrayList<>();
		
		// NIF del perceptor
		// El 190 admite NIF de personas jurdicas en el TIPO2.
		perceptorBean.setValor(PerceptorBeanTipos.P_NIFPER, registro.substring(17, 26).toUpperCase());
		
		// AO NACIMIENTO.
		
		// El modelo 190 admite aos de nacimiento a partir de 1900 y el algoritmo de retenciones los admite
		// a partir de 1905. Para salvar esta discrepancia, si el ao del registro 190 esta entre 1900 y 1905
		// lo sustituyo por 1905.
		String anioPer = registro.substring(152, 156);
		int iAnioPer = Integer.parseInt(anioPer);
		
		if (iAnioPer >= 1900 && iAnioPer < 1905) {
			anioPer = "1905";
		}
		
		perceptorBean.setValor(PerceptorBeanTipos.P_ANIOPER, anioPer);
		
		// CEUTA MELILLA
		final String ceutaMelilla = registro.substring(151, 152);
		if ("1".equals(ceutaMelilla)) {
			perceptorBean.setValor(PerceptorBeanTipos.P_RESICEME, Boolean.TRUE);
			perceptorBean.setValor(PerceptorBeanTipos.P_RENCEME, Boolean.TRUE);
		}
		
		// DISCAPACIDAD
		final int discapacidad = Integer.parseInt(registro.substring(166, 167));
		switch (discapacidad) {
		case 1:
			perceptorBean.setValor(PerceptorBeanTipos.P_DISCAPER, PerceptorBeanTipos.ENTRE33Y65);
			break;
		case 2:
			perceptorBean.setValor(PerceptorBeanTipos.P_DISCAPER, PerceptorBeanTipos.ENTRE33Y65);
			perceptorBean.setValor(PerceptorBeanTipos.P_MOVILPER, Boolean.TRUE);
			break;
		case 3:
			perceptorBean.setValor(PerceptorBeanTipos.P_DISCAPER, PerceptorBeanTipos.MAS65);
			break;
		default:
			break;
		}
		
		// SITUACIN FAMILIAR
		final String situacionFamiliar = registro.substring(156, 157);
		perceptorBean.setValor(PerceptorBeanTipos.P_SITUFAM, situacionFamiliar);
		if ("2".equals(situacionFamiliar)) {
			// NIF DEL CNYUGE
			String nifCon = registro.substring(157, 166).toUpperCase();
			ValidaNif vln = new ValidaNif();
			int resul = vln.checkNif(nifCon);
			if (resul == ValidaNif.NIF_ERROR || !vln.isOk()) {
				// Para poder hacer el clculo no es necesario el NIF del cnyuge.
				// Si el NIF que viene en el registro no es correcto lo cambiamos por uno
				// vlido.
				nifCon = "00000150N";
			}
			perceptorBean.setValor(PerceptorBeanTipos.P_NIFCON, nifCon);
		}
		
		// SITUACIN PERSONAL
		final String clave = registro.substring(77, 78);
		
		if (!"ABC".contains(clave)) {
			perceptorBean.setErr(new ErrorValidacionBeanImpl(TipoErrorValidacion.ERROR, "E1901", "Clave de percepcin no admitida", "", -1));
		} else if ("A".equals(clave)) {
			perceptorBean.setValor(PerceptorBeanTipos.P_SITUPER, 1);
			perceptorBean.setValor(PerceptorBeanTipos.P_CONTRATO, Integer.parseInt(registro.substring(167, 168)));
			// Movilidad geogrfica. Como no existe registro del 190 2017 an, no se de donde extraer este dato.
			// Vamos a suponer que sigue en el mismo sitio que en 2015.
			final String movilidadGeografica = registro.substring(169, 170);
			if ("1".equals(movilidadGeografica)) {
				perceptorBean.setValor(PerceptorBeanTipos.P_MOVIL, Boolean.TRUE);
			}
		} else if ("B".equals(clave)) {
			final String subclave = registro.substring(78, 80);
			
			if ("01".equals(subclave) /*|| "02".equals(subclave)*/) {
				perceptorBean.setValor(PerceptorBeanTipos.P_SITUPER, 2);
			} else if ("03".equals(subclave)) {
				perceptorBean.setValor(PerceptorBeanTipos.P_SITUPER, 4);
			} else {
				perceptorBean.setErr(new ErrorValidacionBeanImpl(TipoErrorValidacion.ERROR, "E1902", "Subclave de percepcin no admitida", "", -1));
			}
		} else {
			perceptorBean.setValor(PerceptorBeanTipos.P_SITUPER, 3);
		}
		
		// RETRIBUCIONES NTEGRAS. La suma de las percepciones dinerarias y en especie. 
		// De momento se ignora el signo porque para retenciones no admite cantidades negativas.
		// Hay que controlarlo o solo van a llamarnos con registros correctos?.
		if ("N".equals(registro.substring(80,81)) || "N".equals(registro.substring(107,108)) 
				|| "N".equals(registro.substring(254,255)) || "N".equals(registro.substring(281,282))) {
			perceptorBean.setErr(new ErrorValidacionBeanImpl(TipoErrorValidacion.ERROR, "E1903", "No se admiten reembolsos", "", -1));
		}
		if (Integer.parseInt(registro.substring(147, 151)) != 0) {
			perceptorBean.setErr(new ErrorValidacionBeanImpl(TipoErrorValidacion.ERROR, "E1904", "No se admiten atrasos", "", -1));
		} else {
			final BigDecimal retrib = new BigDecimal(registro.substring(81, 94)).add(new BigDecimal(registro.substring(108, 121))).
					add(new BigDecimal(registro.substring(255, 268))).add(new BigDecimal(registro.substring(282, 295))).movePointLeft(2);
			perceptorBean.setValor(PerceptorBeanTipos.P_RETRIB, retrib);
		}
		
		// REDUCCIONES POR IRREGULARIDAD
		final BigDecimal irregular = new BigDecimal(registro.substring(170, 183)).movePointLeft(2);
		perceptorBean.setValor(PerceptorBeanTipos.P_IRREGULAR2, irregular);
		
		// COTIZACIONES. 
		final BigDecimal cotizaciones = new BigDecimal(registro.substring(183, 196)).movePointLeft(2);
		perceptorBean.setValor(PerceptorBeanTipos.P_COTIZACIONES, cotizaciones);

		// PENSION COMPENSATORIA EN FAVOR DEL CNYUGE. 
		final BigDecimal conyuge = new BigDecimal(registro.substring(196, 209)).movePointLeft(2);
		perceptorBean.setValor(PerceptorBeanTipos.P_CONYUGE, conyuge);

		// ANUALIDADES. 
		final BigDecimal anualidades = new BigDecimal(registro.substring(209, 222)).movePointLeft(2);
		perceptorBean.setValor(PerceptorBeanTipos.P_ANUALIDADES, anualidades);
		
		// COMUNICACIN PRSTAMOS VIVIENDA
		if ("1".equals(registro.substring(253, 254))) {
			perceptorBean.setValor(PerceptorBeanTipos.P_PRESVIV, Boolean.TRUE);
		}
		
		descendientes(perceptorBean, registro);
		ascencientes(perceptorBean, registro);
		
		if (!erroresRegistro.isEmpty()) {
			for (ErrorValidacionBeanImpl errorValidacionBean : erroresRegistro) {
				perceptorBean.setErr(errorValidacionBean);
			}
		}
	}

	private void descendientes(PerceptorBean perceptorBean, String registro) {
		// Descendientes menores
		final int numDescendietesMenor3 = Integer.parseInt(registro.substring(222, 223));
		final int numDescendientesResto = Integer.parseInt(registro.substring(224, 226));
		
		if (numDescendietesMenor3 + numDescendientesResto == 0) {
			return;
		}
		
		List<DescendienteBean> descendientes = new ArrayList<>();

		// Primero los de tres o ms aos.
		for (int i = 0; i < numDescendientesResto; i++) {
			DescendienteBean bean = perceptorBean.createDescendienteBean();
			bean.setAnioNacimiento(2000);
			descendientes.add(bean);
		}
		
		// Ahora los de menos de tres.
		for (int i = 0; i < numDescendietesMenor3; i++) {
			DescendienteBean bean = perceptorBean.createDescendienteBean();
			bean.setAnioNacimiento(2020);
			descendientes.add(bean);
		}
		
		// Resto de por enteros
		int numRestoPorEntero = Integer.parseInt(registro.substring(226, 228));
		int numMenor3PorEntero = Integer.parseInt(registro.substring(223, 224));
		
		// Asignamos por entero a los tres primeros hijos si los hubiera
		final boolean porEnteroHijo1 = "1".equals(registro.substring(250, 251));
		descendientes.get(0).setPorEntero(porEnteroHijo1);
		if (porEnteroHijo1 && descendientes.get(0).getAnioNacimiento() == 2000) {
			numRestoPorEntero--;
		} else if (porEnteroHijo1) {
			numMenor3PorEntero--;
		}
		
		final int numeroTotalDescendientes = descendientes.size();
		
		if (numeroTotalDescendientes > 1) {
			final boolean porEnteroHijo2 = "1".equals(registro.substring(251, 252));
			descendientes.get(1).setPorEntero(porEnteroHijo2);
			if (porEnteroHijo2 && descendientes.get(1).getAnioNacimiento() == 2000) {
				numRestoPorEntero--;
			} else if (porEnteroHijo2) {
				numMenor3PorEntero--;
			}
		}
		
		if (numeroTotalDescendientes > 2) {
			final boolean porEnteroHijo3 = "1".equals(registro.substring(252, 253));
			descendientes.get(2).setPorEntero(porEnteroHijo3);
			if (porEnteroHijo3 && descendientes.get(2).getAnioNacimiento() == 2000) {
				numRestoPorEntero--;
			} else if (porEnteroHijo3) {
				numMenor3PorEntero--;
			}
		}
		// Resto de por enteros
		int i = 3;
		while (i < numeroTotalDescendientes && (numMenor3PorEntero > 0 || numRestoPorEntero > 0)) {
			DescendienteBean bean = descendientes.get(i);
			
			bean.setPorEntero(true);
			if (bean.getAnioNacimiento() == 2000) {
				numRestoPorEntero--;
			} else {
				numMenor3PorEntero--;
			}
			
			i++;
		}
		
		if (numRestoPorEntero + numMenor3PorEntero > 0) {
			perceptorBean.setErr(new ErrorValidacionBeanImpl(TipoErrorValidacion.ERROR, "E1905", "Existen discrepancias entre el nmero de descendientes y el nmero de descendientes computados por entero.", "", -1));
			return;
		}
		
		
		// DISCAPACITADOS
		int total3365 = Integer.parseInt(registro.substring(228, 230));
		int total3365Porentero = Integer.parseInt(registro.substring(230, 232));
		i = 0;
		//33-65 Por entero
		while (i < numeroTotalDescendientes && total3365Porentero > 0) {
			DescendienteBean bean = descendientes.get(i); 
			if (bean.isPorEntero()) {
				bean.setDiscapacidad(PerceptorBeanTipos.ENTRE33Y65);
				total3365Porentero--;
				total3365--;
			}
			i++;
		}
		// 33-65 Por mitad
		i = 0;
		while (i < numeroTotalDescendientes && total3365 > 0) {
			DescendienteBean bean = descendientes.get(i); 
			if (!bean.isPorEntero()) {
				bean.setDiscapacidad(PerceptorBeanTipos.ENTRE33Y65);
				total3365--;
			}
			i++;
		}

		if (total3365Porentero + total3365 > 0) {
			perceptorBean.setErr(new ErrorValidacionBeanImpl(TipoErrorValidacion.ERROR, "E1906", "Existen discrepancias entre el nmero de descendientes y el nmero de descendientes discapacitados al 33-64%.", "", -1));
			return;
		}
		
		// Movilidad reducida
		int totalMovilidad = Integer.parseInt(registro.substring(232, 234));
		int totalMovilidadPorentero = Integer.parseInt(registro.substring(234, 236));
		// Movilidad por entero
		i = 0;
		while (i < numeroTotalDescendientes && totalMovilidadPorentero > 0) {
			DescendienteBean bean = descendientes.get(i); 
			if (bean.isPorEntero() && bean.getDiscapacidad() == PerceptorBeanTipos.ENTRE33Y65) {
				bean.setMovilidadReducida(true);
				totalMovilidadPorentero--;
				totalMovilidad--;
			}
			i++;
		}
		// Movilidad por mitad
		i = 0;
		while (i < numeroTotalDescendientes && totalMovilidad > 0) {
			DescendienteBean bean = descendientes.get(i); 
			if (!bean.isPorEntero() && bean.getDiscapacidad() == PerceptorBeanTipos.ENTRE33Y65) {
				bean.setMovilidadReducida(true);
				totalMovilidad--;
			}
			i++;
		}
		
		if (totalMovilidadPorentero + totalMovilidad > 0) {
			perceptorBean.setErr(new ErrorValidacionBeanImpl(TipoErrorValidacion.ERROR, "E1907", "Existen discrepancias entre el nmero de descendientes y el nmero de descendientes discapacitados al 33-64% con movilidad reducida.", "", -1));
			return;
		}

		// 65%
		int total65 = Integer.parseInt(registro.substring(236, 238));
		int total65Porentero = Integer.parseInt(registro.substring(238, 240));
		// 65 por entero
		i = 0;
		while (i < numeroTotalDescendientes && total65Porentero > 0) {
			DescendienteBean bean = descendientes.get(i); 
			if (bean.isPorEntero() && bean.getDiscapacidad() == PerceptorBeanTipos.SINDISCAPACIDAD) {
				bean.setDiscapacidad(PerceptorBeanTipos.MAS65);
				total65Porentero--;
				total65--;
			}
			i++;
		}
		// 65 por mitad
		i = 0;
		while (i < numeroTotalDescendientes && total65 > 0) {
			DescendienteBean bean = descendientes.get(i); 
			if (!bean.isPorEntero() && bean.getDiscapacidad() == PerceptorBeanTipos.SINDISCAPACIDAD) {
				bean.setDiscapacidad(PerceptorBeanTipos.MAS65);
				total65--;
			}
			i++;
		}
		
		if (total65Porentero + total65 > 0) {
			perceptorBean.setErr(new ErrorValidacionBeanImpl(TipoErrorValidacion.ERROR, "E1908", "Existen discrepancias entre el nmero de descendientes y el nmero de descendientes discapacitados al 65%.", "", -1));
			return;
		}
		
		//Asignamos los descendientes
		perceptorBean.setValor(PerceptorBeanTipos.P_DESCENDIENTES, descendientes);
	}

	private void ascencientes(PerceptorBean perceptorBean, String registro) {
		final int numMenos75 = Integer.parseInt(registro.substring(240, 241));
		final int numMas75 = Integer.parseInt(registro.substring(242, 243));
		
		if (numMenos75 + numMas75 == 0) {
			return;
		}
		
		List<AscendienteBean> ascendientes = new ArrayList<>();
		
		// Ascendientes de menos de 75 aos.
		int numMenos75Porentero = Integer.parseInt(registro.substring(241, 242));
		for(int i = 0; i < numMenos75; i++) {
			AscendienteBean ascendiente = perceptorBean.createAscendienteBean();
			ascendiente.setAnioNacimiento(PerceptorBeanTipos.ANIOEJ - 65);
			if (numMenos75Porentero > 0) {
				ascendiente.setConvivencia(1);
				numMenos75Porentero--;
			} else {
				// Desde el 190 no disponemos del grado de convivencia
				// de cada descendiente. Este dato es necesario para calcular el mnimo
				// personal y familiar. Esto impide calcular las retenciones tal y como
				// est ahora mismo el algoritmo diseado.
				ascendiente.setConvivencia(2);
			}
			ascendientes.add(ascendiente);
		}
		
		// Ascendientes de 75 aos o ms.
		int numMas75Porentero = Integer.parseInt(registro.substring(243, 244));
		for(int i = 0; i < numMas75; i++) {
			AscendienteBean ascendiente = perceptorBean.createAscendienteBean();
			ascendiente.setAnioNacimiento(PerceptorBeanTipos.ANIOEJ - 76);
			if (numMas75Porentero > 0) {
				ascendiente.setConvivencia(1);
				numMas75Porentero--;
			} else {
				// Desde el 190 no disponemos del grado de convivencia
				// de cada descendiente. Este dato es necesario para calcular el mnimo
				// personal y familiar. Esto impide calcular las retenciones tal y como
				// est ahora mismo el algoritmo diseado.
				ascendiente.setConvivencia(2);
			}
			ascendientes.add(ascendiente);
		}
		
		if (numMenos75Porentero + numMas75Porentero > 0) {
			perceptorBean.setErr(new ErrorValidacionBeanImpl(TipoErrorValidacion.ERROR, "E1909", "Existen discrepancias entre el nmero de ascendientes y el nmero de ascendientes coomputados por entero.", "", -1));
			return;
		}
		
		int numAs3365 = Integer.parseInt(registro.substring(244, 245)); 
		int numAs3365Porentero = Integer.parseInt(registro.substring(245, 246)); 

		int i = 0;
		//33-65 Por entero
		final int numTotalAscendientes = ascendientes.size();
		while (i < numTotalAscendientes && numAs3365Porentero > 0) {
			AscendienteBean bean = ascendientes.get(i); 
			if (bean.getConvivencia() == 1) {
				bean.setDiscapacidad(PerceptorBeanTipos.ENTRE33Y65);
				numAs3365Porentero--;
				numAs3365--;
			}
			i++;
		}
		// 33-65 Por mitad
		i = 0;
		while (i < numTotalAscendientes && numAs3365 > 0) {
			AscendienteBean bean = ascendientes.get(i); 
			if (bean.getConvivencia() > 1) {
				bean.setDiscapacidad(PerceptorBeanTipos.ENTRE33Y65);
				numAs3365--;
			}
			i++;
		}

		if (numAs3365 + numAs3365Porentero > 0) {
			perceptorBean.setErr(new ErrorValidacionBeanImpl(TipoErrorValidacion.ERROR, "E19010", "Existen discrepancias entre el nmero de ascendientes y el nmero de ascendientes discapacitados al 33-64%.", "", -1));
			return;
		}

		// Movilidad reducida
		int totalMovilidad = Integer.parseInt(registro.substring(246, 247));
		int totalMovilidadPorentero = Integer.parseInt(registro.substring(247, 248));
		// Movilidad por entero
		i = 0;
		while (i < numTotalAscendientes && totalMovilidadPorentero > 0) {
			AscendienteBean bean = ascendientes.get(i); 
			if (bean.getConvivencia() == 1 && bean.getDiscapacidad() == PerceptorBeanTipos.ENTRE33Y65) {
				bean.setMovilidadReducida(true);
				totalMovilidadPorentero--;
				totalMovilidad--;
			}
			i++;
		}
		// Movilidad por mitad
		i = 0;
		while (i < numTotalAscendientes && totalMovilidad > 0) {
			AscendienteBean bean = ascendientes.get(i); 
			if (bean.getConvivencia() > 1 && bean.getDiscapacidad() == PerceptorBeanTipos.ENTRE33Y65) {
				bean.setMovilidadReducida(true);
				totalMovilidad--;
			}
			i++;
		}

		if (totalMovilidadPorentero + totalMovilidad > 0) {
			perceptorBean.setErr(new ErrorValidacionBeanImpl(TipoErrorValidacion.ERROR, "E19011", "Existen discrepancias entre el nmero de ascendientes y el nmero de ascendientes discapacitados al 33-64% con movilidad reducida.", "", -1));
			return;
		}

		// 65%
		int total65 = Integer.parseInt(registro.substring(248, 249));
		int total65Porentero = Integer.parseInt(registro.substring(249, 250));
		// 65 por entero
		i = 0;
		while (i < numTotalAscendientes && total65Porentero > 0) {
			AscendienteBean bean = ascendientes.get(i); 
			if (bean.getConvivencia() == 1 && bean.getDiscapacidad() == PerceptorBeanTipos.SINDISCAPACIDAD) {
				bean.setDiscapacidad(PerceptorBeanTipos.MAS65);
				total65Porentero--;
				total65--;
			}
			i++;
		}
		// 65 por mitad
		i = 0;
		while (i < numTotalAscendientes && total65 > 0) {
			AscendienteBean bean = ascendientes.get(i); 
			if (bean.getConvivencia() > 1 && bean.getDiscapacidad() == PerceptorBeanTipos.SINDISCAPACIDAD) {
				bean.setDiscapacidad(PerceptorBeanTipos.MAS65);
				total65--;
			}
			i++;
		}
		
		if (total65Porentero + total65 > 0) {
			perceptorBean.setErr(new ErrorValidacionBeanImpl(TipoErrorValidacion.ERROR, "E19012", "Existen discrepancias entre el nmero de ascendientes y el nmero de ascendientes discapacitados al 65%.", "", -1));
			return;
		}
		
		
		perceptorBean.setValor(PerceptorBeanTipos.P_ASCENDIENTES, ascendientes);
	}

	private void traza(String mensaje) {
		// Activo las trazas solo para desarrollo.
		if (ADHT_JDIT_Factory.getContextoSrv().getENTORNO() == ContextoSrv.ENTORNO_DESARROLLO) {
			final SimpleDateFormat df = new SimpleDateFormat("dd/MM/yyyy HH:mm:ss.S");
			final String laTraza = String.format("c200::CalculoRetencionesCICSSrv [%s]: %s", df.format(new Date()), mensaje);
	
			
			Logger.getLogger(getClass().getName()).log(Level.INFO, laTraza);
		}
	}


}
